package client;

import java.util.Vector;

import javax.microedition.lcdui.*;
import javax.microedition.lcdui.game.GameCanvas;
import javax.microedition.m3g.Appearance;
import javax.microedition.m3g.Graphics3D;
import javax.microedition.m3g.Light;
import javax.microedition.m3g.Loader;
import javax.microedition.m3g.Mesh;
import javax.microedition.m3g.Object3D;
import javax.microedition.m3g.Texture2D;
import javax.microedition.m3g.Transform;
import javax.microedition.m3g.World;
import javax.microedition.m3g.Background;
import javax.microedition.midlet.MIDletStateChangeException;

/**
 * Main game object.
 * 
 * @author Eric
 */
public class Game extends GameCanvas implements Runnable, CommandListener {

	// Main program.
	private MobileGameClient client;

	// Background object.
	private Background background = new Background();

	// Commands.
	private Command quiz = new Command("Take quiz", Command.OK, 0);

	private Command exit = new Command("Exit", Command.EXIT, 1);

	// Display object.
	private static Display display;

	// M3G object.
	private Graphics3D mG3D;

	// Camera.
	public GameCamera gameCam;

	// Game world.
	public World world;

	// Player models.
	private Mesh[] models = new Mesh[2];

	// For FPS calculations
	private boolean fpsEnabled = true;

	private long timeRunning = 0;

	private long timeSinceStart = 0;

	private int framesPerSecond = 0;

	private int fpsCount = 0;

	// Input for the game.
	private GameInput input;

	// private Timer inputTimer;

	// Motion updater for server.
	private MotionUpdateThread motionUpdate;

	// Graphics drawing thread.
	private DrawGraphics drawGraphics;

	// Unread chat messages.
	private boolean unreadChat = false;

	/**
	 * Instantiates game object.
	 * 
	 * @param m
	 * @param d
	 */
	public Game(MobileGameClient m) {
		super(true);
		client = m;
		display = Display.getDisplay(m);
		mG3D = Graphics3D.getInstance();
		background.setColor(0);
		// loadWorld( "/Level/Level.m3g" );
		// loadWorld( "/spacewalls.m3g" );
		loadWorld("/spacewalltwo.m3g");

		// Load models.
		models[0] = loadModel("/man.m3g");
		models[1] = loadModel("/woman.m3g");

		// Thread to draw graphics.
		drawGraphics = new DrawGraphics(this);
		drawGraphics.setPriority(Thread.MIN_PRIORITY);
	}

	/**
	 * Get the game client.
	 * 
	 * @return
	 */
	public MobileGameClient getClient() {
		return client;
	}

	/**
	 * Gets a model from its numerical id. Must call c'tor before this function.
	 * 
	 * @param id
	 * @return
	 */
	public Mesh getModel(long id) {
		return models[(int) id];
	}

	/**
	 * Loads a 3d world.
	 * 
	 * @param filename
	 *            a path to an m3g file (normally in /res/ folder)
	 */
	private void loadWorld(String filename) {
		// Load 3d world.
		try {
			client.debug("Loading " + filename + "...");
			Object3D[] objects = Loader.load(filename);
			client.debug("...loaded!");

			for (int i = 0; i < objects.length; i++) {
				if (objects[i] instanceof World) {
					world = (World) objects[i];
					break;
				}
			}
		} catch (Exception e) {
			System.out.println(e.getMessage());
		}

		// Add a light so the scene is visible.
		Light l = new Light();
		l.setMode(Light.AMBIENT);
		l.setIntensity(.7f);
		world.addChild(l);

		// Disable lighting off for all textures.
		for (int i = 0; i < world.getChildCount(); i++) {
			javax.microedition.m3g.Node n = world.getChild(i);
			if (n instanceof Mesh) {
				Appearance a = ((Mesh) n).getAppearance(0);
				Texture2D t = a.getTexture(0);
				t.setBlending(Texture2D.FUNC_REPLACE);
			}
		}

	}

	/**
	 * Loads a 3d model.
	 * 
	 * @param filename
	 *            a path to an m3g file (normally in /res/ folder)
	 */
	private Mesh loadModel(String filename) {
		Mesh mesh = null;
		// Load 3d model.
		try {
			client.debug("Loading " + filename + "...");
			Object3D[] objects = Loader.load(filename);
			client.debug("...loaded!");

			// Find the "world" in the file.
			World myWorld = null;
			for (int i = 0; i < objects.length; i++) {
				if (objects[i] instanceof World) {
					myWorld = (World) objects[i];
					break;
				}
			}

			// Extract mesh (assume there is only 1.)
			for (int i = 0; i < myWorld.getChildCount(); i++) {
				if (myWorld.getChild(i) instanceof Mesh) {
					mesh = (Mesh) myWorld.getChild(i);
					break;
				}
			}
			myWorld.removeChild(mesh);

		} catch (Exception e) {
			client.log(e.getMessage());
			return mesh;
		}

		return mesh;
	}

	/**
	 * Handle commands.
	 */
	public void commandAction(Command c, Displayable d) {
		if (c.equals(exit)) {
			try {
				client.destroyApp(false);
			} catch (MIDletStateChangeException e) {
				client.log("Could not terminate application");
			}
		} else if (c.equals(quiz)) {
			input.deactivate();
			motionUpdate.deactivate();
			drawGraphics.deactivate();
			client.startQuiz();
		}
	}

	/**
	 * Prepares to end the program.
	 */
	public void cleanup() {
		// inputTimer.cancel();
		drawGraphics.kill();
	}

	/**
	 * Draw FPS meter.
	 * 
	 * @param g
	 */
	private void drawFps(Graphics g) {

		// Keep times.
		timeRunning += System.currentTimeMillis() - timeSinceStart;
		timeSinceStart = System.currentTimeMillis();

		// Reset when we hit one second.
		if (timeRunning >= 1000) {
			framesPerSecond = fpsCount + 1;
			fpsCount = 0;
			timeRunning = 0;
		}

		// Draw FPS on screen.
		g.setColor(255, 255, 255);
		fpsCount++;
		g.drawString("FPS: " + framesPerSecond, 0, 0, Graphics.TOP
				| Graphics.LEFT);
	}

	/**
	 * Sets whether or not to show unread chat message.
	 * 
	 * @param unread
	 */
	public void setUnreadChat(boolean unread) {
		unreadChat = unread;
	}

	/**
	 * Draw unread chat message
	 * 
	 * @param g
	 */
	private void drawChatUnread(Graphics g) {
		// Draw FPS on screen.
		g.setColor(255, 255, 255);
		g
				.drawString("Unread chat message", 0, 10, Graphics.TOP
						| Graphics.LEFT);
	}

	/**
	 * Render the game.
	 * 
	 * @param g
	 */
	private void draw() {
		long time = client.currentTime();
		// if ( lastDraw == -1 ) lastDraw = time;
		Graphics g = getGraphics();
		try {
			// Update user models.
			Vector users = client.getUserList();
			Transform t;
			for (int i = 0; i < users.size(); i++) {
				User u = (User) users.elementAt(i);
				float motionFactor = (float) (time - u.getLastMotionTime());// lastDraw
																			// );
																			// //
																			// should
																			// be
																			// last
																			// update!!!
				if (motionFactor < 0f)
					motionFactor = 0f;

				// Get transform matrix.
				t = new Transform(u.getTransform());

				// Move.
				float fSpeed;
				if (motionFactor == 0f)
					fSpeed = 0;
				else
					fSpeed = u.getForwardSpeed() * motionFactor;
				t.postTranslate(0f, 0f, fSpeed);

				// Rotate.

				if (u.getId() == client.getUserId()) {
					t.postRotate(u.getYAxisAngularVelocity() * motionFactor,
							0f, 1f, 0f);
					gameCam.setTransform(t);
				} else {
					t.postRotate(u.getYAxisAngularVelocity() * motionFactor,
							0f, 1f, 0f);
					u.getModel().setTransform(t);
				}

			}
			// lastDraw = client.currentTime();

			// Draw world.
			mG3D.bindTarget(g, true, Graphics3D.ANTIALIAS
					| Graphics3D.TRUE_COLOR | Graphics3D.DITHER);
			mG3D.clear(background);
			world.setActiveCamera(gameCam);
			mG3D.render(world);
		} catch (Exception e) {
			System.err.println(e.getMessage());
		} finally {
			mG3D.releaseTarget();
		}

		// FPS
		if (fpsEnabled) {
			drawFps(g);
		}

		// Unread chat messages.
		if (unreadChat) {
			drawChatUnread(g);
		}

		// end of drawing code
		flushGraphics();

	}

	/**
	 * Start the 3d world.
	 */
	public void start() {
		// Camera object.
		gameCam = new GameCamera(this, client.getMyUser());

		input = new GameInput(this);
		input.activate();
		input.start();
		display.setCurrent(this);

		// Thread for retrieving motion updates.
		motionUpdate = new MotionUpdateThread(client);
		motionUpdate.start();

		// Start (or restart) graphics thread.
		if (!drawGraphics.isAlive()) {
			drawGraphics.start();
		}

		// Framerate counter.
		timeSinceStart = System.currentTimeMillis();

		// Command listeners.
		addCommand(exit);
		addCommand(quiz);
		setCommandListener(this);

		// Start thread.
		Thread runner = new Thread(this);
		runner.start();
	}

	/**
	 * Show this screen.
	 */
	public void show() {
		input.activate();
		motionUpdate.activate();
		drawGraphics.activate();
		display.setCurrent(this);
	}

	/**
	 * Continuously updates the game's graphics.
	 */
	public void run() {
		// Create graphics drawing thread.
		drawGraphics.activate();
	}

	/**
	 * Motion speed.
	 * 
	 * @return
	 */
	public float getClientSpeed() {
		// TODO: don't hard-code this!!!
		return 0.002f;
	}

	/**
	 * Rotational speed.
	 * 
	 * @return
	 */
	public float getClientAngularSpeed() {
		// TODO: don't hard-code this!!!
		return 0.05f;
	}

	public void motionUpdate() {
		motionUpdate.update();
	}

	/**
	 * Thread for drawing graphics. Set at a low priority to prevent graphics
	 * from taking too many resources.
	 * 
	 * @author Eric
	 */
	class DrawGraphics extends Thread {

		// Game canvas.
		private Game game;

		// Toggles the graphics.
		private boolean active = false;

		// Used to end the thread entirely.
		private boolean alive = true;

		public DrawGraphics(Game game) {
			super();
			this.game = game;
		}

		/**
		 * Turn on graphics.
		 */
		public void activate() {
			active = true;
		}

		/**
		 * Turn off graphics.
		 */
		public void deactivate() {
			active = false;
		}

		/**
		 * Ends the thread.
		 */
		public void kill() {
			alive = false;
			deactivate();
		}

		/**
		 * Runs the thread (call start(), NOT this function!)
		 */
		public void run() {

			// Activation loop.
			while (alive) {

				// Graphics loop.
				while (active) {
					game.draw();
					// Ensure there's leftover time!
					try {
						Thread.sleep(5);
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}

				// Don't waste CPU when not drawing.
				// Wait patiently for thread to activate.
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}

	}

	/**
	 * Show a chat window.
	 */
	public void showChat() {
		input.deactivate();
		motionUpdate.deactivate();
		drawGraphics.deactivate();
		client.showChat();
	}

}
