package client;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Vector;

import javax.microedition.io.Connector;
import javax.microedition.io.SocketConnection;
import javax.microedition.m3g.Mesh;
import javax.microedition.midlet.MIDlet;
import javax.microedition.midlet.MIDletStateChangeException;

import client.request.ClientRequest;
import client.request.ClientRequestGlobalUpdate;
import client.request.ClientRequestLogin;
import client.request.ClientRequestLogout;
import client.request.ClientRequestNextQuestion;
import client.request.ClientRequestQuiz;
import client.response.ClientResponse;
import client.screens.Chat;
import client.screens.Login;

public class MobileGameClient extends MIDlet {
	
	private String host = "localhost";
	private int port = 7654;
	
	// 3d game canvas.
	public Game canvas = new Game( this );
	
	// Chat screen.
	public Chat chat = new Chat( this );
	
	// Users logged in.
	private Vector userList = new Vector();
	private Hashtable userTable = new Hashtable();
	
	// Teams in the game.
	private Vector teamList = new Vector();
	private Hashtable teamTable = new Hashtable();
	
	// This client's user object.
	private User myUser = null;
	
	// Socket.
	protected SocketConnection socket = null;
	
	// Data streams.
	protected DataInputStream dataIn = null;
	protected DataOutputStream dataOut = null;
	
	// Time (in ms) between movement updates.
	// Tweak this parameter for each device.
	private long updateSpeed = 10;
	
	// When true, debug messages are enabled.
	// Warning: this generates a LOT of messages!
	private boolean debugMode = false;
	
	// Thread for reading in and executing responses.
	private ResponseThread responseThread = null;
	
	// Used to kill the client by ResponseDisconnect.
	private boolean active = true; 
	
	
	/**
	 * C'tor.
	 */
	public MobileGameClient() {
		super();
	}
	
	
	
	/**
	 * Gets the data input stream for reading.
	 * @return valid input stream or null.
	 */
	public DataInputStream getInputStream() {
		return dataIn;
	}
	
	
	/**
	 * Gets the data output stream for reading.
	 * @return valid output stream or null.
	 */
	public DataOutputStream getOutputStream() {
		return dataOut;
	}
	
	
	/**
	 * Connect to the server and get valid I/O streams.
	 * @return the socket object.
	 */
	protected void connect() throws IOException {
		if ( null == socket ) {
			// Open socket
			log( "Opening connection on " + host + " port " + port );
			socket = (SocketConnection) Connector.open( "socket://" + host + ":" + port );
			socket.setSocketOption( SocketConnection.DELAY, 0 );
			socket.setSocketOption( SocketConnection.KEEPALIVE, 0 );
			dataIn = new DataInputStream( socket.openDataInputStream() );
			dataOut = new DataOutputStream( socket.openDataOutputStream() );
		}
	}
	
	/**
	 * Sends a request to the server.
	 * @param req
	 * @throws IOException
	 */
	public void sendRequest( ClientRequest req ) throws IOException {
		req.setClient( this );
		req.run();
	}
	
	
	/**
	 * Helper function to get time.
	 * @return time in MS since epoch format.
	 */
	public long currentTime() {
		return System.currentTimeMillis();
	}


	/**
	 * Used for frequent debug logging messages.
	 * Use debugMode boolean to toggle these
	 * messages.
	 * @param msg
	 */
	public void debug( String msg ) {
		if ( debugMode ) log( msg );
	}



	public void destroyApp( boolean unconditional ) throws MIDletStateChangeException {
		debug( "Destroy app called with: " + unconditional );
		
		// TODO:
		//if ( !unconditional ) {
			// Throw up a screen here.
		//}
		
		
		// Kill the graphics.
		canvas.cleanup();
		
		try {
			// Tell server we're through.
			ClientRequestLogout req = new ClientRequestLogout();
			sendRequest( req );
			
			// Wait for ClientResponseDisconnected to tell
			// us to diaf and/or gtfo.
			int i = 0;
			while ( active ) {
				if ( i > 10 ) break; // don't wait forever
				try {
					Thread.sleep( 100 );
				} catch ( InterruptedException e ) {
					log( e.getMessage() );
				}
				i++;
			}
			
			// Stop getting updates.
			responseThread.deactivate();
			
			log( "Disconnected!" );
		} catch ( IOException e ) {
			log( e.getMessage() );
		} finally {
			notifyDestroyed();
		}
	}

	public void pauseApp() {
		// TODO Auto-generated method stub
		
	}

	protected void login() {
		Login login = new Login( this );
		login.start();
		debug("Exiting from start.");
	}
	
	
	/**
	 * Attempt to login a player with the following user/pass combo.
	 * @param username
	 * @param password
	 */
	public void login( String username, String password ) {
		// Attempt to login.
		ClientRequestLogin req = new ClientRequestLogin();
		req.setUsername( username );
		req.setPassword( password );
		try {
			sendRequest( req );
		} catch ( IOException e ) {
			log( "Could not send login: " + e.getMessage() );
		}
	}
	
	
	// Make the 3d graphics start.
	public void start3dWorld() {
		this.myUser.setLastMotionTime( currentTime() );
		canvas.start();
	}
	
	
	public void startQuiz() {
		debug("Trying to start quiz...");
		ClientRequestQuiz quiz = new ClientRequestQuiz();
		try {
			sendRequest( quiz );
		} catch (IOException e) {
			log( "StartQuiz could not send request." );
			e.printStackTrace();
		}		
	}
	
	/**
	 * Send a request to get the next question.
	 */
	public void nextQuestion() {
		// Send request.
		ClientRequestNextQuestion next = new ClientRequestNextQuestion();
		try {
			sendRequest( next );
		} catch (IOException e) {
			log( "nextQuestion could not send request." );
			e.printStackTrace();
		}
	}

	/**
	 * Logs an error message.
	 * @param message
	 */
	public void log( String message ) {
		System.out.println( message );
	}
	

	public void startApp() throws MIDletStateChangeException {
		
		// Init response interpreter table.
		ClientResponseTable.init();
		
		// Connect to server.
		try {
			connect();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		// Start getting responses.
		responseThread = new ResponseThread( this );
		responseThread.start();
		
		// Create chat window.
		chat.start();
		
		// Login.
		login();
	}
	
	/**
	 * Shows the chat window.
	 * Used for chat/3d toggle.
	 */
	public void showChat() {
		chat.show();
	}
	
	/**
	 * Add a chat message.
	 * @param name
	 * @param message
	 */
	public void addToChat( String name, String message ) {
		chat.addChatText( name, message );
	}
	
	/**
	 * Shows the 3d world.
	 * Used for chat/3d toggle.
	 */
	public void show3dWorld() {
		canvas.show();
	}
	

	/**
	 * Adds a user to the game.
	 * @param u user to add.
	 */
	public void addUser( User u ) {
		userList.addElement( u );
		userTable.put( new Long( u.getId() ), u );
	}
	
	/**
	 * Lookup a user.
	 * @param
	 */
	public User getUser( long userId ) {
		return (User)userTable.get( new Long(userId) );
	}
	
	public void removeUser( User u ) {
		userList.removeElement( u );
		userTable.remove( new Long( u.getId() ) );
	}
	
	
	/**
	 * Returns list of users.
	 * @return
	 */
	public Vector getUserList() {
		return userList;
	}

	/**
	 * Adds a team object to the game.
	 * @param t
	 */
	public void addTeam( Team t ) {
		teamList.addElement( t );
		teamTable.put( new Long( t.getId() ), t );
	}
	
	/**
	 * Retrieves a team object based on its ID number.
	 * May return null if invalid.
	 * @param teamId a valid team id number.
	 * @return
	 */
	public Team getTeam( long teamId ) {
		return (Team) teamTable.get( new Long( teamId ) );
	}
	
	/**
	 * Used by ResponseDisconnect to officially kill
	 * the client after we've disconnected.
	 */
	public void deactivate() {
		active = false;
	}


	/**
	 * Closes the socket connection.
	 */
	public void closeSocket() {
		try {
			socket.close();
			socket = null;
		} catch (IOException e) {
			e.printStackTrace();
		}		
	}
	
	/**
	 * Gets the client user object.
	 * @return
	 */
	public User getMyUser() {
		return myUser;
	}
	
	/**
	 * Sets the client user object.
	 * @param u
	 */
	public void setMyUser( User u ) {
		myUser = u;
	}
	
	/**
	 * Tells canvas if there's an unread chat message or not.
	 * @param unread true if there's an unread message, else set false.
	 */
	public void setUnreadChat( boolean unread ) {
		canvas.setUnreadChat( unread );
	}

	/**
	 * Tweakable speed sets the maximum time between
	 * sending motion updates to the server.
	 * @return
	 */
	public long getUpdateSpeed() {
		return updateSpeed;
	}

	/**
	 * Gets the id of the client user.
	 * @return user id, or -1 if not logged in.
	 */
	public long getUserId() {
		if ( null == myUser ) {
			return -1;
		} else {
			return myUser.getId();
		}
	}

	/**
	 * Resume the game after a break of some
	 * kind, either a quiz, test, etc. or if
	 * the user has *gasp* recieved a ye olde 
	 * telephone call.
	 */
	public void resumeGame() {
		// Ask for an update.
//		try {
//			ClientRequestGlobalUpdate req = new ClientRequestGlobalUpdate();
//			sendRequest( req );
//		} catch ( IOException e ) {
//			log( e.getMessage() );
//		}
		
		// Resume game canvas.
		canvas.start();
	}
	
	/**
	 * Gets model from its numerical id.
	 * @param id
	 * @return
	 */
	public Mesh getModel( long id ) {
		return canvas.getModel( id );
	}
	
	/**
	 * When the game is running, we must handle all
	 * responses from the server.  This thread handles
	 * that task.
	 * @author Eric
	 */
	class ResponseThread extends Thread {
		
		private MobileGameClient client = null;
		
		private boolean active = true;
		
		/**
		 * C'tor.
		 * @param client a valid MobileGameClient.
		 */
		public ResponseThread( MobileGameClient client ) {
			super();
			this.client = client;
		}
		
		/**
		 * Kills this thread.
		 */
		public void deactivate() {
			active = false;
		}
		
		/**
		 * Runs the thread.
		 * DO NOT CALL THIS METHOD!  Call start() instead!!!
		 */
		public void run() {
			while ( active ) {
				try {
					int id = client.getInputStream().readInt();
					client.debug( "getResponse just got id = " + id );
					ClientResponse resp = ClientResponseTable.get( id );
					resp.setClient( client );
					resp.run();
				} catch (IOException e) {
					log( e.getMessage() );
				} catch (InstantiationException e) {
					log( e.getMessage() );
				} catch (IllegalAccessException e) {
					log( e.getMessage() );
				} catch (ClassNotFoundException e) {
					log( e.getMessage() );
				}
			}
		} // run()
		
	} // ResponseThread

	
}
