package core;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Vector;
import core.request.GameRequest;
/**
 * Abstract class for the game server.
 * @author eric
 */
public abstract class GameServer {
	
	// Socket used by the server for incoming requests.
	private ServerSocket serverSocket;
	
	private int numThreads = 50;
	
	// Work queue.
	private WorkQueue work;
	
	// All sectors.
	protected HashMap<Long, Sector> sectors = new HashMap<Long, Sector>();
	
	// All teams, maintained in both map and vector.
	private HashMap<Long, Team> teamMap = new HashMap<Long, Team>();
	private Vector<Team> teamList = new Vector<Team>();
	
	// Activates debug mode. Used for generating very specific log
	// files.  WARNING: enabling this will generate a ridiculous
	// amount of logs!!!
	private boolean debugMode = false;
	
	
	
	/**
	 * Creates the DB object.
	 * @return
	 */
	protected abstract void createDB() throws SQLException;
	
	/**
	 * Returns the DB object.
	 * @return
	 */
	protected abstract GameDB getDb();
	
	/**
	 * Initializes the server.  Call this before start() to get
	 * the server ready to go.
	 */
	public void init() {
		log( "Server init" );
		
		// Connect to database.
		try {
			createDB();
		} catch (SQLException e1) {
			log( "Could not connect to db:" );
			log ( e1.getMessage() );
			System.exit( -5 );
		}
		
		// Start work queue;
		work = new WorkQueue( numThreads );
		
		// Fetch and init teams from DB.
		getDb().getTeams();
		
		// Create request name table.
		GameRequestTable.init();
		
		// Start motion timer.
		//motionTimer.scheduleAtFixedRate( motionTimerTask, 0, getUpdateSpeed() );
	}
	
	/**
	 * Starts the game server, listens for TCP requests.
	 */
	public void start() {
		log( "Starting up" );
		Socket clientSocket = null;
		// Bind to socket.
		try {
		    serverSocket = new ServerSocket( getPortNumber() );
		    log( "Server started and listening on port: " + getPortNumber() );
		} catch (IOException e) {
		    log( "Could not listen on port: " + getPortNumber() ); 
		    System.exit(-1); // Fatal error.
		}
		
		// Enter server loop.
		while( true ) {
			try {
				// Get socket and read into stream.
			    clientSocket = serverSocket.accept();
			    
			    // Pass to a new work thread.
			    SocketReaderThread th = new SocketReaderThread( this, clientSocket );
			    th.start();
				
			} catch (IOException e) {
				log( e.getMessage() ); 
			}
		}
	}
	
	/**
	 * Prints log message.  This function can
	 * be changed to direct messages to a logfile.
	 * @param msg the log message.
	 */
	public void log( String msg ) {
		System.out.println( msg );
	}
	
	/**
	 * Prints debug messages if debugMode is true.
	 * @param msg the log message.
	 */
	public void debug( String msg ) {
		if ( debugMode ) log( msg );
	}
	
	/**
	 * Helper function to get time.
	 * @return time in MS since epoch format.
	 */
	public long currentTime() {
		return System.currentTimeMillis();
	}
	
	/**
	 * Specifies the number of threads to create.
	 * @return number of threads, must be > 0
	 */
	public abstract int getNumThreads();
	
	
	/**
	 * Specifies the port number.
	 * @return port number, must be valid and not in use.
	 */
	public abstract int getPortNumber();
	
	/**
	 * Logs in a user.
	 * @param username
	 * @param password
	 * @return user Id if okay, -1 if otherwise.
	 */
	public abstract long login( String username, String password );
	
	
	/**
	 * Signs up a user.
	 * @param userid
	 */
	public abstract void signup( String username, String password );
	
	
	/**
	 * Adds a user to a team.
	 * @param userId
	 * @param teamId
	 */
	public abstract void joinTeam( long userId, long teamId );
	
	/**
	 * Removes a user from a team.
	 * @param userId
	 * @param teamId
	 */
	public abstract void leaveTeam( long userId, long teamId );
	
	/**
	 * Spawn a user into their appropriate sector, location,
	 * and heading.
	 * @param u
	 */
	public abstract void spawnUser( User u );
	
	
	/**
	 * Set a user value to a specific number.
	 * @param userId id of user.
	 * @param value
	 */
	public void setUserValue( String valueName, long userId, long value ) {
		getUser( userId ).setValue( valueName, value );
		getDb().setUserValue( valueName, userId, value ); 
	}
	
	public void addUserValue( String valueName, long userId, long value ) {
		getUser( userId ).increaseValue( valueName, value );
		getDb().increaseUserValue( valueName, userId, value );
	}
	
	public void decreaseUserValue( String valueName, long userId, long value ) {
		getUser( userId ).decreaseValue( valueName, value );
		getDb().decreaseUserValue( valueName, userId, value );		
	}
	
	/**
	 * Returns a team based on its numerical id.
	 * @param teamId numerical (DB) id of team.
	 * @return team object, or null if it does not exists.
	 */
	public Team getTeam( long teamId ) {
		return teamMap.get( teamId );
	}
	
	/**
	 * Returns a user based on its numerical id.
	 * @param userId	numerical id of user
	 * @return user object, or null if it does not exist.
	 */
	public abstract User getUser( long userId );

	public void userAnimation( long userId, long animationId, GameRequest req ) {
		// Get the user
		User myUser = this.getUser(userId);
		try {			
			// Here this user need to update itself on Server side
			// by entering its current status to DB				
			myUser.updateAnimation(animationId);			
			// Maybe here need to deal with DB?			
		} catch (Exception e) {
			log("Error Animation: In User updating.");
			e.printStackTrace();
		}
	}

	public void userEnterSector( long userId, long sectorId, GameRequest req ) {
		// Get the user
		User myUser = this.getUser(userId);
		// Find out which sector the user enters
		Sector eSector = sectors.get(sectorId);
		try {			
			// Here this user need to update itself on Server side
			// by entering its current status to DB				
			myUser.changeSector(eSector);
			// Maybe here need to deal with DB?			
		} catch (Exception e) {
			log("Error Sector enter: In User updating.");
			e.printStackTrace();
		}
	}

	public void userLeaveSector(long userId, long sectorId, GameRequest req) {
		// Get the user
		User myUser = this.getUser(userId);
		// Find out which sector the user leaves
		Sector lSector = sectors.get(sectorId);
		try {			
			// Here this user need to update itself on Server side
			// by entering its current status to DB				
			myUser.changeSector(null);
			lSector.removeUser(myUser);
			// Maybe here need to deal with DB?			
		} catch (Exception e) {
			log("Error Sector enter: In User updating.");
			e.printStackTrace();
		}
	}
	

	/**
	 * Return vector of all teams.
	 * @return
	 */
	public Vector<Team> getTeamList() {
		return teamList;
	}
	
	/**
	 * Adds a team.  Used for initialization of server
	 * and for when a user creates a new team.
	 * 
	 * Note that this will *NOT* add the team to the DB!!  The
	 * team must already exist in the DB before this function
	 * is called.
	 * 
	 * @param t a valid team object, must already have
	 * valid id.
	 */
	public void addTeam( Team t ) {
		teamMap.put( t.getId(), t );
		teamList.add( t );
	}
	
	/**
	 * Removes a user.
	 */
	public abstract void removeUser( User u );
	
	public abstract Vector<core.User> getUserList();

	/**
	 * Gets the work queue object for this server.
	 * @return
	 */
	public WorkQueue getWorkQueue() {
		return work;
	}

	
}
