package core;


import java.io.OutputStream;
import java.util.Date;
import java.util.HashMap;

/**
 * Holds information about logged-in players such as
 * their user id, name, team, location, etc. 
 * 
 * @author eric
 *
 */
public class User {
	
	// Name of the character.
	private String name = "";
	
	// User id.
	private long id = -1;
	
	// Values (points, cash, etc.)
	private HashMap<String, Long> value = new HashMap<String, Long>();
	
	// Team, or null if no team.
	private Team team = null;
	
	// Location/rotation.
	//private Transform3D transform = new Transform3D();
	
	float[] transformMatrix = new float[16];
	
	// Forward and sideways speed.
	private float speedForward = 0f;
	private float speedSideways = 0f;
	private float xAxisAngularVelocity = 0f;
	private float yAxisAngularVelocity = 0f;
	private float zAxisAngularVelocity = 0f;
	
	// The output stream for this user.	
	private OutputStream outputStream = null;
	
	
	// When the last motion happened.
	// Time is reported by client.
	private long lastMotionUpdateTime = -1;
	
	// Local area of user.
	private Sector sector = null; 
	
	// ID of character model.
	private long modelId = 0;
	
	// ID of animation, or 0 if standing still.
	private long animation = 0;
	
	// When the current animation was started.
	private Date animationStartTime = null;
		
	// When the most recent request was sent for
	// this user.  This is used *only* for checking
	// for a timeout.
	private Date lastRequestTime = null;
	
	// True only when the user is interested in ResponseUpdates,
	// otherwise false.  If false, no ResponseUpdates will be sent.
	private boolean isUpdating = false;	
	
	public User() {
		// Set transform to an identity matrix.
		int i = 0;
		for ( int y = 0; y < 4; y++ ) {
			for ( int x = 0; x < 4; x++ ) {
				if ( x == y ) {
					transformMatrix[i] = 1;
				} else {
					transformMatrix[i] = 0;
				}
				i++;
			}
		}
		
	}
	
	/**
	 * Sets the name.
	 * @param name name of character (username)
	 */
	public void setName( String name ) {
		this.name = name;
	}
	
	/**
	 * Sets the id.
	 * @param id id of character
	 */
	public void setId( long id ) {
		this.id = id;
	}
	
	/**
	 * Gets the ID number of this character.
	 * @return id number, or -1 if invalid.
	 */
	public long getId() {
		return id;
	}
	
	/**
	 * Gets the name of this character.
	 * @return name, or the empty string if invalid.
	 */
	public String getName() {
		return name;
	}
	
	/**
	 * Gets a stored value from this character
	 * @param name name of value
	 * @return value, or -1 if not present.
	 */
	public long getValue( String name ) {
		if ( value.containsKey( name ) ) {
			return value.get( name );
		} else {
			return -1;
		}
	}
	
	/**
	 * Sets a value to a specific number.  BE CAREFUL
	 * with this, setting a value in this way could
	 * result in unpredictable behavior.
	 * 
	 * @param name name of new or existing value
	 * @param newValue the new value to set it to 
	 */
	public void setValue( String name, long newValue ) {
		value.put( name, newValue );
	}
	
	/**
	 * Increases a value.  No overflow detection.
	 * 
	 * @param name name of the value
	 * @param amount the amount to increase the value
	 */
	public void increaseValue( String name, long amount ) {
		long v = value.get( name );
		value.put( name, v + amount );
	}
	
	/**
	 * Decreases a value.  No overflow is detected, no
	 * < 0 is checked for.
	 * 
	 * @param name name of the value
	 * @param amount the amount to decrease the value
	 */
	public void decreaseValue( String name, long amount ) {
		long v = value.get( name );
		value.put( name, v - amount );
	}

	
	/**
	 * Set this user to a given team.
	 * @param team the user's team, or null if no team.
	 */
	public void setTeam( Team team ) {
		this.team = team;
	}

	/**
	 * Fetch this user's team.
	 * @return the user's team, or null if no team.
	 */
	public Team getTeam() {
		return team;
	}


	public void setModelId( long modelID ) {
		this.modelId = modelID;
	}

	public long getModelId() {
		return modelId;
	}
	
	public Sector getSector(){
		return sector;
	}
	
	public void setSector( Sector s ) {
		sector = s;
	}
	
	public void updateAnimation(long animationId) {
		animation = animationId;		
	}
	
	public Date getAniTime(){
		return animationStartTime;
	}
	
	public void setAniTime(Date aTime){
		animationStartTime = aTime;
	}

	public void changeSector(Sector sector) {
		this.sector = sector;		
	}

	public void setLastRequestTime(Date lastRequestTime) {
		this.lastRequestTime = lastRequestTime;
	}

	public Date getLastRequestTime() {
		return lastRequestTime;
	}
	
	/**
	 * Turns ResponseUpdates off.
	 */
	public void turnUpdatesOff() {
		isUpdating = false;
	}
	
	/**
	 * Turns ResponseUpdates on.
	 */
	public void turnUpdatesOn() {
		isUpdating = true;
	}
	
	/**
	 * True if update mode is on.
	 * @return
	 */
	public boolean isInUpdateMode() {
		return isUpdating;
	}

	/**
	 * Retrieves the transformation matrix.
	 * @return transformation matrix (16 floats)
	 */
	public float[] getTransform() {
		return this.transformMatrix;
		//this.transform.get( transformMatrix );

//		for (int i = 0; i < 16; i++) {
//			System.out.println( i+ "  " +transformMatrix[i]);
//		}
//		System.out.println();
	}
	
	/**
	 * Sets the user's transform matrix.
	 * @param transformMatrix
	 */
	public void setTransform( float[] transformMatrix ) {
		//transform.set( transformMatrix );
		this.transformMatrix = transformMatrix;
	}

	/**
	 * Gets forward speed.
	 * @return forward speed
	 */
	public float getForwardSpeed() {
		return speedForward;
	}

	/**
	 * Gets sideways speed.
	 * @return sideways speed
	 */
	public float getSidewaysSpeed() {
		return speedSideways;
	}
	
	/**
	 * Sets forward speed.
	 * @param f
	 */
	public void setForwardSpeed( float f ) {
		speedForward = f;
	}
	
	/**
	 * Sets sideways speed.
	 * @param s
	 */
	public void setSidewaysSpeed( float s ) {
		speedSideways = s;
	}
	
	/**
	 * Gets the angular velocity on x axis.
	 */
	public float getXAxisAngularVelocity() {
		return xAxisAngularVelocity;
	}	
	
	/**
	 * Set the angular velocity on x axis.
	 * @param x
	 */
	public void setXAxisAngularVelocity( float x ) {
		xAxisAngularVelocity = x;
	}	
	
	/**
	 * Gets the angular velocity on y axis.
	 */
	public float getYAxisAngularVelocity() {
		return yAxisAngularVelocity;
	}	
	
	/**
	 * Set the angular velocity on y axis.
	 * @param y
	 */
	public void setYAxisAngularVelocity( float y ) {
		yAxisAngularVelocity = y;
	}
	
	/**
	 * Gets the angular velocity on z axis.
	 */
	public float getZAxisAngularVelocity() {
		return zAxisAngularVelocity;
	}	
	
	/**
	 * Set the angular velocity on z axis.
	 * @param z
	 */
	public void setZAxisAngularVelocity( float z ) {
		zAxisAngularVelocity = z;
	}	
	

	/**
	 * Remembers when the last motion update happened.
	 * @param updateTime
	 */
	public void setLastMotionTime( long updateTime ) {
		lastMotionUpdateTime = updateTime;
	}
	
	/**
	 * Gets the last time the user's motion was updated, or
	 * -1 if it has not been udpated.
	 * @return
	 */
	public long getLastMotionUpdateTime() {
		return lastMotionUpdateTime;
	}
	
	public void setOutputStream( OutputStream out ) {
		outputStream = out;
	}

	public OutputStream getOutputStream() {
		return this.outputStream;
	}
	
		
}
