package core.request;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;
import java.util.Date;

import core.GameServer;
import core.User;

/**
 * Request object used by WorkQueue.  Create version of this
 * class for each request type.
 * 
 * @author eric
 */
public abstract class GameRequest implements Runnable {
	
	// Socket from client request.
	protected Socket socket;
	
	// Reference to game server.
	protected GameServer gs;
		
	// Input stream from socket.
	private InputStream in = null;
	
	private DataInputStream dataRead = null;
	
	// User sending this request.
	protected User user;
	
	// Time when this request comes.
	protected Date timestamp;
	
	/**
	 * Sets the value of this request from a
	 * client socket.
	 * @param s client socket
	 * @throws IOException
	 */
	public void set( Socket s, InputStream in, GameServer gs) {
		this.socket = s;
		this.in = in;
		this.gs = gs;
		timestamp = new Date(); // Defaults to current time.
	}
	
	/**
	 * Parses the request.  This function should only read in
	 * data from the inputstream.  The run() function will
	 * setup and execute the request.
	 */
	public abstract void parse();
	
	/**
	 * Runs this request.  Note that parse will be called
	 * before run.
	 */
	public abstract void run();
	
	/**
	 * Gets the socket associated with this request.
	 * @return socket from with which request was sent.
	 */
	public Socket getSocket() {
		return socket;
	}
	
	public User getUser() {
		return user;
	}
	

	/**
	 * Gets a data input stream for internal use.
	 * @return valid DataInputStream or null
	 */
	protected DataInputStream getData() {
		if ( dataRead == null ) {
			dataRead = new DataInputStream( in );
		}
		return dataRead;
	}
	
	/**
	 * Get the timestamp of this request
	 * @return		the timestamp
	 */
	public Date getTimestamp() {
		return timestamp;
	}
}
