package spaceGame;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import spaceGame.response.ResponseQuestion;
import spaceGame.response.ResponseQuestionChoice;

import core.GameServer;
import core.Team;
import spaceGame.User;

/**
 * Handles DB functions used by Space Game.
 * @author eric
 *
 */
public class GameDB extends core.GameDB {

	public GameDB( GameServer gs ) throws SQLException {
		super(gs);
	}
	
	/**
	 * Validates a user
	 * @param name username
	 * @param password password
	 * @return new User object, or null if name/pass is invalid.
	 */
	public spaceGame.User validateUser( String name, String password ) {
		ResultSet rs = null;
        PreparedStatement ps = null;
        try {
            ps = conn.prepareStatement( "SELECT * FROM users WHERE username = ? AND password = ?" );
            ps.setString( 1, name );
            ps.setString( 2, password );
            rs = ps.executeQuery();
            
            // If we can't access the row, the username/password is
            // invalid so return null.
            if ( !rs.next() ) {
                return null;
            }
            
            // Create a User object and populate with the results.
            spaceGame.User u = new spaceGame.User();
            u.setName( name );
            u.setId( rs.getInt( "user_id" ) );
            u.setModelId( rs.getInt( "model_id" ) );
            long teamID = rs.getInt( "team_id" );
            Team team = server.getTeam( teamID );
            u.setTeam( team );
            team.addMember( u );
            
            return u;
            
        } catch ( Exception e ) {
        	server.log( e.getMessage() );
        }
        
        // Should never get here (error).
        return null;
	}
	
	public ResponseQuestion getQuizQuestionFor( User u ) {
		ResultSet rs = null;
        PreparedStatement ps = null;
        ResponseQuestion resp;
        try {
            ps = conn.prepareStatement( "SELECT * FROM quiz_questions " +
            		"LEFT OUTER JOIN quiz_questions_answered " +
            		"ON quiz_questions.quiz_question_id = quiz_questions_answered.quiz_question_id " +
            		"WHERE quiz_questions_answered.user_id IS NULL " +
            		"ORDER BY RAND() LIMIT 1" );
            rs = ps.executeQuery();
            
            // If we can't access the row, the username/password is
            // invalid so return null.
            if ( !rs.next() ) {
                return null;
            }
            
            long qid = rs.getLong( "quiz_question_id" );
            u.setLastQuestionId( qid );
            String questionText = rs.getString( "question" );
            
            // For now, assume all questions are multiple choice.
            // TODO allow for other types of responses.
            resp = new ResponseQuestionChoice();
            resp.set( server );
            ((ResponseQuestionChoice)resp).set( qid, questionText );
            
            // Get choices.
            ps = conn.prepareStatement( "SELECT * FROM quiz_choices WHERE quiz_question_id = ? ORDER BY choice_id ASC" );
            ps.setLong( 1, qid );
            rs = ps.executeQuery();
            while ( rs.next() ) {
            	String choice = rs.getString( "choice" );
            	char id = rs.getString( "choice_id" ).charAt( 0 );
            	((ResponseQuestionChoice)resp).addChoice( id, choice );
            }
            
            
            return resp;
            
        } catch ( Exception e ) {
        	server.log( e.getMessage() );
        }
        
        // Should never get here (error).
        return null;
	}

	/**
	 * Checks a mutliple choice question in the database.
	 * @param questionId id o
	 * @param choice
	 * @return
	 */
	public boolean checkQuizQuestionChoice( long questionId, char choice ) {
		ResultSet rs = null;
        PreparedStatement ps = null;
        try {
            ps = conn.prepareStatement( "SELECT answer FROM quiz_questions WHERE quiz_question_id = ?" );
            ps.setLong( 1, questionId );
            rs = ps.executeQuery();
            
            // If we can't access the row, the question is invalid!
            if ( !rs.next() ) {
                return false;
            }
            
            char answer = rs.getString( "answer" ).charAt( 0 );
            return ( answer == choice ); 
            
        } catch ( Exception e ) {
        	server.log( e.getMessage() );
        }
        
        // Should never get here (error).
        return false;
	}
	
	
	/**
	 * Note that the user has answered this question, and whether
	 * they got it correct.
	 * @param userId
	 * @param questionId
	 * @param correct
	 */
	public void quizQuestionAnswered( long userId, long questionId, boolean correct ) {
        PreparedStatement ps = null;
        try {
        	// TODO: handle questions which have already been answered ("important") questions.
        	// Since these may already have an answer, we can't simply use an insert.
        	
        	// Insert row into quiz_questions_answered.
            ps = conn.prepareStatement( 
            		"INSERT INTO quiz_questions_answered " +
            		"(quiz_question_id, user_id, was_correct) VALUES (?, ?, ?)" );
            ps.setLong( 1, questionId );
            ps.setLong( 2, userId );
            ps.setBoolean( 3, correct );
            int updates = ps.executeUpdate();
            server.debug( "quizQuestionAnswered, " + updates + " rows affected." );
            
        } catch ( Exception e ) {
        	server.log( e.getMessage() );
        }
	}
	
	

}
